/** @file   utils.cpp
 * @brief   Implementation of Utils class
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/13 16:29:02 $
 * @author  Tomi Lamminsaari
 */


#include "utils.h" // class's header file
#include "gameobject.h"
#include "warglobals.h"
using eng2d::Vec2D;
using std::string;
using std::istream;

namespace WeWantWar {


/** Calculates the angle of a vector from 'origo' to 'p'.
 */
fixed Utils::findVecAngle( const Vec2D& origo, const Vec2D& p )
{
  Vec2D dirVec = p - origo;
  fixed a = 0;
  
  if ( dirVec.y() < 0 ) {
    float cosine = (-dirVec.x()) / dirVec.length();
    a = fixacos( ftofix(cosine) );
    a -= itofix( 64 );

  } else {
    float cosine = dirVec.x() / dirVec.length();
    a = fixacos( ftofix(cosine) );
    a -= itofix( 192 );
  }
  return a;
}



/** Calculates the turning direction.
 */
int Utils::findTurningDir( const eng2d::Vec2D& from, const eng2d::Vec2D& to,
                           int fromAngle )
{
  Vec2D dirVec = to - from;

  Vec2D refpoint1( 0, -dirVec.length() );
  Vec2D refpoint2( 0, -dirVec.length() );
  Vec2D refpoint3( 0, -dirVec.length() );

  refpoint1.rotate( fromAngle - 2 );
  refpoint2.rotate( fromAngle );
  refpoint3.rotate( fromAngle + 2 );
  refpoint1 += from;
  refpoint2 += from;
  refpoint3 += from;

  float refDist1 = (to - refpoint1).length();
  float refDist2 = (to - refpoint2).length();
  float refDist3 = (to - refpoint3).length();

  if ( refDist1 < 10 && refDist2 < 10 && refDist3 < 10 ) {
    return 0;
  }
  if ( refDist1 < refDist2 && refDist1 < refDist3 ) {
    return -1;
  }
  if ( refDist2 < refDist1 && refDist2 < refDist3 ) {
    return 0;
  }
  return 1;
}



/** Tells if 'pObj1' is facing towards the 'pObj2'
 */
bool Utils::isFacingObject( const GameObject* pObj1, const GameObject* pObj2,
                            float factor )
{
  Vec2D refpoint( 0, -(objectDistance(pObj1, pObj2)) );
  refpoint.rotate( pObj1->angle() );
  refpoint += pObj1->position();
  float refDist = ( pObj2->position() - refpoint ).length();
  if ( refDist < factor ) {
    return true;
  }
  return false;
}



/** Tells if 'pObj' is facing the coordinate
 */
bool Utils::isFacingCoordinate( const GameObject* pObj, const Vec2D& pos,
                                float factor )
{
  Vec2D distVec( pos );
  distVec -= pObj->position();
  
  Vec2D refpoint( 0, -(distVec.length()) );
  refpoint.rotate( pObj->angle() );
  refpoint += pObj->position();
  
  float refDist = ( pos - refpoint ).length();
  if ( refDist < factor ) {
    return true;
  }
  return false;
}



/** Calculates the distance between the two GameObjects.
 */
float Utils::objectDistance( const GameObject* pObj1, const GameObject* pObj2 )
{
  Vec2D tmpVec1( pObj1->position() );
  tmpVec1 -= pObj2->position();
  return tmpVec1.length();
}



/** Checks if the two objects can see each other.
 */
bool Utils::objectsSeeEachOther( const GameObject* pObj1,
                                 const GameObject* pObj2,
                                 float maxDist )
{
  float objDist = objectDistance( pObj1, pObj2 );
  if ( objDist > maxDist ) {
    return false;
  }
  
  Vec2D dirV( pObj2->position() );
  dirV -= pObj1->position();
  dirV.norm();
  dirV *= 16;
  
  int numOfRounds = static_cast<int>( objDist / 16 );
  
  Vec2D tmpPos( pObj1->position() );
  for ( int i=0; i < numOfRounds; i++ ) {
    if ( Map::collide( tmpPos ) == true ) {
      return false;
    }
    tmpPos += dirV;
  }
  return true;
}



/** Tells if living creatures are inside a circle
 */
bool Utils::creatureInsideCircle( const eng2d::Vec2D& rP, float radius )
{
  for ( int i=0; i < WarGlobals::pObjTable->objectList.size(); i++ ) {
    GameObject* pO = WarGlobals::pObjTable->objectList.at(i);

    if ( pO->state() == GameObject::STATE_LIVING ) {
      if ( pO->objectType() != ObjectID::TYPE_CAR ) {
        eng2d::Vec2D distVec( pO->position() );
        distVec -= rP;
        if ( distVec.length() < radius ) {
          return true;
        }
      }
    }
  }
  return false;
}



/** Tells if there is a car inside the inspection circle
 */
GameObject* Utils::carInsideCircle( const eng2d::Vec2D& rP, float radius )
{
  for ( int i=0; i < WarGlobals::pObjTable->vehicleObjects.size(); i++ ) {
    GameObject* pO = WarGlobals::pObjTable->vehicleObjects.at(i);

    if ( pO->state() == GameObject::STATE_LIVING ) {
      if ( pO->objectType() == ObjectID::TYPE_CAR ) {
        eng2d::Vec2D distVec( pO->position() );
        distVec -= rP;
        if ( distVec.length() < radius ) {
          return pO;
        }
      }
    }
  }
  return 0;
}



/** Searches for given string from stream.
 */
bool Utils::searchForString( istream& rIn, const string& searchFor )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return true;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == searchFor ) {
      return false;
    }
  }
}



/** A predator blender function. The default transparency blender with one
 * little modification.
 */
unsigned long Utils::predator_blend_b16( unsigned long x, unsigned long y,
                                         unsigned long n )
{
  unsigned long result;

  if (n)
    n = (n + ( rand() % 32 ) ) / 8;

  x = ((x & 0xFFFF) | (x << 16)) & 0x7E0F81F;
  y = ((y & 0xFFFF) | (y << 16)) & 0x7E0F81F;

  result = ((x - y) * n / 32 + y) & 0x7E0F81F;

  return ((result & 0xFFFF) | (result >> 16));
}



bool Utils::pointInsidePolygon(const std::vector<eng2d::Vec2D>& aVertices,
                               const eng2d::Vec2D& aPoint)
{
  int i=0;
	int j=0;
	int c=0;
	bool isInside = false;
	for (i=0, j = aVertices.size()-1; i < aVertices.size(); j = i++) {
		Vec2D vertexI = aVertices[i];
		Vec2D vertexJ = aVertices[j];
		if ((((vertexI.vy <= aPoint.vy) && (aPoint.vy < vertexJ.vy)) ||
		     ((vertexJ.vy <= aPoint.vy) && (aPoint.vy < vertexI.vy))) &&
		    (aPoint.vx < (vertexJ.vx - vertexI.vx) * (aPoint.vy - vertexI.vx) / (vertexJ.vy - vertexI.vy) + vertexI.vx)) {
		    	isInside = !isInside;
		}
	}
	return isInside;
}



} // end of namespace
